<?php
/*
Plugin Name: Clean TP Share Zip
Plugin URI: https://wp-security.jp/clean-tp-share-zip
Description: Export selected themes and plugins to a ZIP file and generate a shareable link with expiration.
Version: 1.0
Author: KOGMA
Author URI: https://kogma.work/
License: GPLv2 or later
Text Domain: clean-tp-share-zip
*/

if (!defined('ABSPATH')) {
    exit;
}

add_action('plugins_loaded', function () {
    load_plugin_textdomain('clean-tp-share-zip', false, dirname(plugin_basename(__FILE__)) . '/languages');
});

define('CTPS_SHARE_DIR', wp_upload_dir()['basedir'] . '/sharezips');
define('CTPS_SHARE_URL', wp_upload_dir()['baseurl'] . '/sharezips');

add_action('admin_menu', function () {
    add_menu_page(
        'Clean TP Share Zip',
        'Clean TP Share Zip',
        'manage_options',
        'ctps_share',
        'ctps_render_page',
        'dashicons-share',
        80
    );
});

function ctps_check_permission() {
    if (get_current_user_id() !== 1) {
        wp_die(esc_html(__('Only the first administrator can use this feature.', 'clean-tp-share-zip')));
    }
}

function ctps_render_page() {
    ctps_check_permission();

    echo '<div class="wrap"><h1>Clean TP Share Zip</h1>';
    echo '<form method="post">';
    wp_nonce_field('ctps_generate_zip', 'ctps_nonce');

    echo '<h2>' . esc_html(__('Themes', 'clean-tp-share-zip')) . '</h2>';
    $themes = wp_get_themes();
    foreach ($themes as $slug => $theme) {
        $dir = $theme->get_stylesheet();
        echo '<label><input type="checkbox" name="ctps_items[]" value="' . esc_attr("theme:$dir") . '"> ' . esc_html($theme->get('Name')) . '</label><br>';
    }

    echo '<h2>' . esc_html(__('Plugins', 'clean-tp-share-zip')) . '</h2>';
    $plugins = get_plugins();
    foreach ($plugins as $path => $plugin) {
        $slug = dirname($path);
        echo '<label><input type="checkbox" name="ctps_items[]" value="' . esc_attr("plugin:$slug") . '"> ' . esc_html($plugin['Name']) . '</label><br>';
    }

    echo '<p>' . esc_html(__('Days to keep:', 'clean-tp-share-zip')) . ' <input type="number" name="ctps_days" value="3" min="1"> ' . esc_html(__('days', 'clean-tp-share-zip')) . '</p>';
    echo '<input type="submit" name="ctps_submit" class="button button-primary" value="' . esc_attr(__('Generate Shareable Link', 'clean-tp-share-zip')) . '">';
    echo '</form></div>';

    if (
        isset($_SERVER['REQUEST_METHOD']) &&
        $_SERVER['REQUEST_METHOD'] === 'POST' &&
        isset($_POST['ctps_submit'])
    ) {
        if (
            !isset($_POST['ctps_nonce']) ||
            !check_admin_referer('ctps_generate_zip', 'ctps_nonce')
        ) {
            wp_die(esc_html(__('Security check failed.', 'clean-tp-share-zip')));
        }

        $selected_raw = isset($_POST['ctps_items']) ? wp_unslash($_POST['ctps_items']) : [];
        $selected = is_array($selected_raw) ? array_map('sanitize_text_field', $selected_raw) : [];
        $days = isset($_POST['ctps_days']) ? intval($_POST['ctps_days']) : 0;

        if (!empty($selected) && $days > 0) {
            $zipname = ctps_generate_zip($selected);
            $expire = time() + $days * DAY_IN_SECONDS;
            update_option('ctps_share_' . $zipname, $expire);

            echo '<div class="updated"><p>';
            echo esc_html(__('Shareable link:', 'clean-tp-share-zip')) . ' <a href="' . esc_url(CTPS_SHARE_URL . '/' . $zipname) . '" id="ctps-link">' . esc_html($zipname) . '</a> ';
            echo esc_html(__('(Expires:', 'clean-tp-share-zip')) . ' ' . esc_html(gmdate('Y-m-d H:i', $expire)) . ') ';
            echo '<button class="button" onclick="navigator.clipboard.writeText(document.getElementById(\'ctps-link\').href)">' . esc_html(__('Copy', 'clean-tp-share-zip')) . '</button>';
            echo '</p></div>';
        } else {
            echo '<div class="error"><p>' . esc_html(__('Please select themes/plugins and specify valid expiration days.', 'clean-tp-share-zip')) . '</p></div>';
        }
    }
}

function ctps_generate_zip($items) {
    $filename = 'ctps-' . gmdate('Ymd-His') . '-' . wp_generate_password(6, false) . '.zip';
    $zip_path = CTPS_SHARE_DIR . '/' . $filename;

    if (!file_exists(CTPS_SHARE_DIR)) {
        wp_mkdir_p(CTPS_SHARE_DIR);
    }

    $zip = new ZipArchive();
    if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
        return '';
    }

    foreach ($items as $item) {
        list($type, $slug) = explode(':', $item);
        $base_dir = $type === 'theme' ? get_theme_root() : WP_PLUGIN_DIR;
        $dir = $base_dir . '/' . $slug;

        if (!is_dir($dir)) {
            continue;
        }

        $folder_prefix = $type === 'theme' ? 'themes/' : 'plugins/';
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($files as $file) {
            $file_path = $file->getRealPath();
            $relative_path = $folder_prefix . $slug . '/' . substr($file_path, strlen($dir) + 1);
            $zip->addFile($file_path, $relative_path);
        }
    }

    $zip->close();
    return $filename;
}

add_action('ctps_cleanup_event', 'ctps_cleanup');

register_activation_hook(__FILE__, function () {
    if (!wp_next_scheduled('ctps_cleanup_event')) {
        wp_schedule_event(time(), 'daily', 'ctps_cleanup_event');
    }
});

register_deactivation_hook(__FILE__, function () {
    wp_clear_scheduled_hook('ctps_cleanup_event');
});

function ctps_cleanup() {
    if (!file_exists(CTPS_SHARE_DIR)) {
        return;
    }

    foreach (glob(CTPS_SHARE_DIR . '/*.zip') as $file) {
        $filename = basename($file);
        $option_key = 'ctps_share_' . $filename;
        $expire = get_option($option_key);
        if ($expire && time() > $expire) {
            wp_delete_file($file);
            delete_option($option_key);
        }
    }
}
?>
