// Clean Lightbox - Vanilla JS
document.addEventListener('DOMContentLoaded', function () {
  // -------- Collect image links (case-insensitive, allow query strings) --------
  const candidateLinks = Array.from(document.querySelectorAll('a[href]')).filter(a => {
    try {
      const href = a.getAttribute('href') || '';
      return /\.(jpe?g|png|webp)(\?.*)?$/i.test(href);
    } catch (_) { return false; }
  });

  // Automatically add .lightbox class and default gallery
  candidateLinks.forEach((a) => {
    if (!a.classList.contains('lightbox')) a.classList.add('lightbox');
    if (!a.dataset.gallery) a.dataset.gallery = 'default';
  });

  const allLinks = candidateLinks;

  // -------- Shared state --------
  let overlay, modal, content, captionBox;
  let currentGroup = [];
  let currentIndex = 0;

  function showLightbox(index, gallery) {
    currentGroup = allLinks.filter(l => (l.dataset.gallery || 'default') === gallery);
    const link = currentGroup[index];
    if (!link) return;
    currentIndex = index;

    // --- Build DOM ---
    overlay = document.createElement('div');
    overlay.className = 'lightbox-overlay';
    overlay.style.opacity = 0;
    overlay.setAttribute('role', 'dialog');
    overlay.setAttribute('aria-modal', 'true');
    overlay.tabIndex = -1;

    modal = document.createElement('div');
    modal.className = 'lightbox-modal';
    modal.addEventListener('click', (e) => e.stopPropagation());

    const wrapper = document.createElement('div');
    wrapper.className = 'lightbox-wrapper';

    content = document.createElement('img');
    content.className = 'lightbox-content';
    content.draggable = false;
    content.addEventListener('dragstart', (e) => e.preventDefault());
    wrapper.appendChild(content);

    captionBox = document.createElement('div');
    captionBox.className = 'lightbox-caption-box';
    wrapper.appendChild(captionBox);

    const controls = document.createElement('div');
    controls.className = 'lightbox-controls';

    const closeBtn = document.createElement('button');
    closeBtn.type = 'button';
    closeBtn.className = 'lightbox-close';
    closeBtn.setAttribute('aria-label', 'Close');
    closeBtn.textContent = '×';
    closeBtn.addEventListener('click', closeLightbox);

    const prevBtn = document.createElement('button');
    prevBtn.type = 'button';
    prevBtn.className = 'lightbox-prev';
    prevBtn.setAttribute('aria-label', 'Previous image');
    prevBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="36" height="36" viewBox="0 0 24 24" aria-hidden="true"><path fill="currentColor" d="M15.41 7.41 10.83 12l4.58 4.59L14 18l-6-6 6-6z"/></svg>`;
    prevBtn.addEventListener('click', (e) => { e.stopPropagation(); navigate(-1); });

    const nextBtn = document.createElement('button');
    nextBtn.type = 'button';
    nextBtn.className = 'lightbox-next';
    nextBtn.setAttribute('aria-label', 'Next image');
    nextBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="36" height="36" viewBox="0 0 24 24" aria-hidden="true"><path fill="currentColor" d="M8.59 16.59 13.17 12 8.59 7.41 10 6l6 6-6 6z"/></svg>`;
    nextBtn.addEventListener('click', (e) => { e.stopPropagation(); navigate(1); });

    controls.appendChild(closeBtn);
    controls.appendChild(prevBtn);
    controls.appendChild(nextBtn);

    modal.appendChild(controls);
    modal.appendChild(wrapper);
    overlay.appendChild(modal);

    // Prevent double tap zoom / dblclick zoom on touch devices
    function killDoubleTap(el) {
      let last = 0;
      el.addEventListener('touchend', (e) => {
        const now = Date.now();
        if (now - last < 350) { e.preventDefault(); }
        last = now;
      }, { passive: false });
      el.addEventListener('dblclick', (e) => e.preventDefault());
    }
    killDoubleTap(overlay);
    killDoubleTap(modal);
    killDoubleTap(content);

    document.body.appendChild(overlay);

    // Close when clicking outside modal
    overlay.addEventListener('click', function (e) {
      if (e.target === overlay) closeLightbox();
    });

    // Lock body scroll
    document.body.classList.add('clb-lock');

    // Prevent scroll/touchmove on background
    const preventScroll = (e) => e.preventDefault();
    overlay.addEventListener('touchmove', preventScroll, { passive: false });
    overlay.addEventListener('wheel', preventScroll, { passive: false });

    // Fade in
    setTimeout(() => {
      overlay.style.transition = 'opacity 0.3s ease';
      overlay.style.opacity = 1;
      overlay.focus();
    }, 10);

    // --- Keyboard navigation (close, prev/next) ---
    const escListener = (e) => {
      if (e.key === 'Escape') closeLightbox();
      if (e.key === 'ArrowLeft') navigate(-1);
      if (e.key === 'ArrowRight') navigate(1);
    };
    document.addEventListener('keydown', escListener);

    // --- Swipe / drag navigation ---
    let isDragging = false;
    let startX = 0;
    let dragMoved = false;

    content.addEventListener('pointerdown', (ev) => {
      if (ev.button !== undefined && ev.button !== 0) return; // Only left button
      isDragging = true;
      dragMoved = false;
      startX = ev.clientX;
      content.classList.add('is-dragging');
      try { content.setPointerCapture(ev.pointerId); } catch (_) {}
    });

    content.addEventListener('pointermove', (ev) => {
      if (!isDragging) return;
      ev.preventDefault();
      if (Math.abs(ev.clientX - startX) > 5) dragMoved = true;
    });

    const endDrag = (ev) => {
      if (!isDragging) return;
      isDragging = false;
      content.classList.remove('is-dragging');
      const dx = ev.clientX - startX;
      const THRESHOLD = 50;
      if (dx > THRESHOLD) navigate(-1);
      else if (dx < -THRESHOLD) navigate(1);
      try { content.releasePointerCapture(ev.pointerId); } catch(_) {}
    };
    content.addEventListener('pointerup', endDrag);
    content.addEventListener('pointercancel', () => {
      isDragging = false;
      content.classList.remove('is-dragging');
    });

    // Click left/right half of image to navigate
    function handleWrapperClick(e) {
      if (e.target.closest('.lightbox-controls') || e.target.closest('.lightbox-caption-box')) return;
      if (dragMoved) return;
      const rect = content.getBoundingClientRect();
      const x = e.clientX, y = e.clientY;
      if (x < rect.left || x > rect.right || y < rect.top || y > rect.bottom) return;
      if (x > rect.left + rect.width / 2) navigate(1);
      else navigate(-1);
    }
    wrapper.addEventListener('click', handleWrapperClick);

    // Cursor hint on hover (resize left/right)
    content.addEventListener('mousemove', (e) => {
      const r = content.getBoundingClientRect();
      content.style.cursor = (e.clientX > r.left + r.width / 2) ? 'e-resize' : 'w-resize';
    });
    content.addEventListener('mouseleave', () => { content.style.cursor = ''; });

    // --- Navigation update & content update ---
    function updateNavVisibility() {
      prevBtn.style.display = (currentIndex === 0) ? 'none' : 'block';
      nextBtn.style.display = (currentIndex === currentGroup.length - 1) ? 'none' : 'block';
    }

    function navigate(step) {
      const newIndex = currentIndex + step;
      if (newIndex >= 0 && newIndex < currentGroup.length) {
        currentIndex = newIndex;
        updateContent(currentGroup[newIndex]);
        updateNavVisibility();
      }
    }

    function cleanup() {
      document.removeEventListener('keydown', escListener);
      overlay.removeEventListener('touchmove', preventScroll);
      overlay.removeEventListener('wheel', preventScroll);
      wrapper.removeEventListener('click', handleWrapperClick);
    }

    function closeLightbox() {
      cleanup();
      if (overlay && overlay.parentNode) overlay.parentNode.removeChild(overlay);
      document.body.classList.remove('clb-lock');
    }

    updateContent(link);
    updateNavVisibility();
  }

  // -------- Update content (image + caption) --------
  function updateContent(link) {
    if (!content || !captionBox) return;

    content.classList.remove('is-ready');

    // Load image
    const src = link.href;
    const img = new Image();
    img.decoding = 'async';
    img.referrerPolicy = 'no-referrer';
    img.src = src;

    img.onload = () => {
      content.src = src;
      requestAnimationFrame(() => {
        content.classList.add('is-ready');
      });
      preloadAround(currentIndex);
    };

    // Reset captions
    captionBox.innerHTML = '';

    // 1) Gutenberg: <figure><figcaption>
    const figure = link.closest('figure');
    if (figure) {
      const caps = figure.querySelectorAll('figcaption');
      caps.forEach(el => {
        const clone = el.cloneNode(true);
        clone.classList.add('lightbox-caption');
        captionBox.appendChild(clone);
      });
    }

    // 1.5) Theme/Swiper: sibling <figcaption> inside slide container
    if (!captionBox.hasChildNodes()) {
      const slideContainer =
        link.closest('.swiper-slide') ||
        (link.closest('.slide-media') ? link.closest('.slide-media').parentElement : null);

      if (slideContainer) {
        const sibCap =
          slideContainer.querySelector('figcaption.slide-title') ||
          slideContainer.querySelector('figcaption');
        if (sibCap) {
          const clone = sibCap.cloneNode(true);
          clone.classList.add('lightbox-caption');
          captionBox.appendChild(clone);
        }
      }
    }

    // 2) Classic editor: .wp-caption .wp-caption-text
    if (!captionBox.hasChildNodes()) {
      const wpCap = link.closest('.wp-caption');
      if (wpCap) {
        const t = wpCap.querySelector('.wp-caption-text');
        if (t) {
          const clone = t.cloneNode(true);
          clone.classList.add('lightbox-caption');
          captionBox.appendChild(clone);
        }
      }
    }

    // 3) Fallback: data-caption / title / img.alt
    if (!captionBox.hasChildNodes()) {
      const viaData = link.getAttribute('data-caption');
      const viaTitle = link.getAttribute('title');
      const imgEl = link.querySelector('img');
      const viaAlt = imgEl ? imgEl.getAttribute('alt') : '';
      const text = viaData || viaTitle || viaAlt || '';
      if (text) {
        const p = document.createElement('div');
        p.className = 'lightbox-caption';
        // Use textContent for safety (avoid XSS)
        p.textContent = text;
        captionBox.appendChild(p);
      }
    }
  }

  // Preload neighbors
  function preloadAround(idx) {
    [idx - 1, idx + 1].forEach(i => {
      if (i >= 0 && i < currentGroup.length) {
        const pre = new Image();
        pre.decoding = 'async';
        pre.src = currentGroup[i].href;
      }
    });
  }

  // -------- Bind click handlers --------
  allLinks.forEach(link => {
    link.addEventListener('click', function (e) {
      e.preventDefault();
      const gallery = link.dataset.gallery || 'default';
      const group = allLinks.filter(l => (l.dataset.gallery || 'default') === gallery);
      const index = group.indexOf(link);
      showLightbox(index, gallery);
    });
  });
});
